#include <iostream>
#include <cstdio>
#include <vector>
#include <map>
#include <algorithm>
#include <stdarg.h>

using namespace std;

#define T_NUM   1000
#define T_LABEL 2000
#define T_END   9999

#define CMD_POP   1
#define CMD_PUSH  2
#define CMD_DUP   3
#define CMD_STORE 4

#define CMD_ADD   5
#define CMD_SUB   6
#define CMD_VADD  7

#define CMD_IF    10
#define CMD_GOTO  11

#define CMD_VDUP   12
#define CMD_VSTORE 13

#define CMD_PRINT 20
#define CMD_HALT  21

#define CMD_LABEL 100
#define CMD_DEBUG 999

struct tcmd {
    int cmd;
    int arg;
    
    tcmd(int c,int a) : cmd(c),arg(a) {};
};

int sp=0;
vector<struct tcmd> program;
vector<int> stack;

int die(const char *fmt, ...) {
  va_list fmt_args;
  va_start(fmt_args,fmt);
  fprintf(stderr,"BLAD: ");
  vfprintf(stderr,fmt,fmt_args);
  va_end(fmt_args);
  fprintf(stderr,"\n");
  exit(1);
}

int usage() {
    fprintf(stderr,"./stos.e PROGRAM STOS\n");
    exit(1);
}

// -----------------------------

char yylval[256];

int get_token(FILE *f) {
    int c=0,n;
    /* bez bialych znakow */
    yylval[0]=0;
    
    if (feof(f)) return T_END;
    
    while (!feof(f)) {
	c=fgetc(f);
	if (c=='#') { // komentarze
	    while (!feof(f)) {
		c=fgetc(f);
		if (c=='\n') break;
	    }
	    return T_END;
	}
	if (c!=' ' && c!='\t' && c!=13) break;
    }
    
    if (c=='\n' || c==EOF) return T_END;
    
    yylval[0]=c;n=1;
    while (!feof(f)) {
	c=fgetc(f);
	if (c==' ' || c=='\t' || c=='\n' || c==13 || c==EOF) {
	    ungetc(c,f);
	    break;
	}
	yylval[n]=c;
	n++;
	if (n==sizeof(yylval)-1) break;
    }
    yylval[n]=0;

    if ((yylval[0]>='0' && yylval[0]<='9') || (yylval[0]=='-' && n>1)) return T_NUM;
    if (yylval[0]=='@' && n>1) return  T_LABEL;   
        
    if (strcmp(yylval,"PUSH")==0) return CMD_PUSH;
    if (strcmp(yylval,"POP")==0) return CMD_POP;
    if (strcmp(yylval,"DUP")==0) return CMD_DUP;
    if (strcmp(yylval,"STORE")==0) return CMD_STORE;
    if (strcmp(yylval,"VDUP")==0) return CMD_VDUP;
    if (strcmp(yylval,"VSTORE")==0) return CMD_VSTORE;
    if (strcmp(yylval,"ADD")==0) return CMD_ADD;
    if (strcmp(yylval,"SUB")==0) return CMD_SUB;
    if (strcmp(yylval,"GOTO")==0) return CMD_GOTO;
    if (strcmp(yylval,"IF")==0) return CMD_IF;
    if (strcmp(yylval,"HALT")==0) return CMD_HALT;
    if (strcmp(yylval,"PRINT")==0) return CMD_PRINT;
    if (strcmp(yylval,"DEBUG")==0) return CMD_DEBUG;
    
    die("nieznany token: %s",yylval);
    return -1;
}

void load_program(const char *filename) {
    map<string,int> labels;
    map<int,string> jumps;
    FILE *f;
    int i,n,d_value;
    
    f=fopen(filename,"r");
    if (f==NULL) die("nie mozna otworzyc: %s",filename);
    while(!feof(f)) {
	int token=get_token(f);
//	printf("token=%d s=%s\n",token,yylval);
	switch(token) {
        case T_END: break;
	case T_LABEL: 
	    n=program.size();
	    program.push_back(tcmd(CMD_LABEL,n));
	    labels[string(yylval)]=n;
	    break; 
	case T_NUM: die("zly token, oczekiwano polecenia, wczytano liczbe: %s",yylval); break;
	
	// polecenia bez argumentow
	case CMD_PRINT: 
	case CMD_DEBUG:
	case CMD_VDUP:
	case CMD_VSTORE:
	case CMD_HALT:
	case CMD_SUB:
	    if (get_token(f)!=T_END) die("oczekiwano konca wiersza, wczytano: %s",yylval);
	    program.push_back(tcmd(token,0));
	    break;
	
	// polecenia z obowiazkowymi argumentami
	case CMD_PUSH:
	case CMD_STORE:
	    i=get_token(f);
	    if (i!=T_NUM) die("oczekiwano liczby, wczytano: %s",yylval);
	    program.push_back(tcmd(token,atoi(yylval)));	    
	    break;	    	    	    	    
	case CMD_IF:
	case CMD_GOTO:
	    i=get_token(f);
	    if (i!=T_LABEL) die("oczekiwano etykiety, wczytano: %s",yylval);

	    n=program.size();	    
	    program.push_back(tcmd(token,-1));	    	    
	    jumps[n]=string(yylval);
	    
	    break; 
	    
	// polecenia z opcjonalnymi argumentami
	case CMD_POP:
	case CMD_DUP:
	case CMD_ADD:
	    d_value=1;
	    i=get_token(f);
	    if (i==T_NUM) d_value=atoi(yylval);
	    else {
		if (token==CMD_ADD) token=CMD_VADD;
		if (i!=T_END) die("oczekiwano liczby lub konca wiersza, wczytano: %s",yylval);
	    }
	
	    program.push_back(tcmd(token,d_value));
	    break;
	    
	default:
	    die("nieznany token: %s (%d)",yylval,token);
	}	
    }
    fclose(f);

/*    
    for(i=0;i<program.size();i++) {
	printf("[%02d] %d, %d\n",i,program[i].cmd,program[i].arg);
    }
*/
    
    // poprawianie skokow
    for (map<int,string>::iterator it=jumps.begin();it!=jumps.end();++it) {
	map<string,int>::iterator f;
	f=labels.find(it->second);
	if (f==labels.end()) {
	    die("brak deklaracji etykiety: %s",it->second.c_str());	    
	} else {
	    program[it->first].arg=f->second;
	}
    }

/*
    for(i=0;i<program.size();i++) {
	printf("[%02d] %d, %d\n",i,program[i].cmd,program[i].arg);
    }
    
    printf("koniec!\n");
*/
}

void load_stack(const char *filename) {
    FILE *f;
    
    f=fopen(filename,"r");
    if (f==NULL) die("nie mozna otworzyc: %s",filename);
    while(!feof(f)) {
	int token=get_token(f);
	switch(token) {
	case T_END:
	    break;
	case T_NUM:
	    stack.push_back(atoi(yylval));
	    break;
	default:
	    die("stos, oczekiwano liczby, wczytano: %s",yylval);
	}
    }
    fclose(f);    
}


//------------------------------

int pop(int x=1) {
    int res;
    if (x<1) die("pop: zla wartosc x=%d",x);
    if (x>sp) die("za malo elementow na stosie x=%d, sp=%d",x,sp);
    
    res=stack[sp-x];
    for(int i=sp-x;i<sp-1;i++) {
	stack[i]=stack[i+1];
    }
    sp--;
    return res;
}

void push(int x) {
    if (sp==(int) stack.size()) {
	stack.push_back(x);
    } else {
	stack[sp]=x;
    }
    sp++;
}

void dup(int x=1) {
    if (x<1) die("dup: zla wartosc x=%d",x);
    if (x>sp) die("za malo elementow na stosie x=%d, sp=%d",x,sp);
    
    push(stack[sp-x]);
}

void store(int x=1) {
    if (x<1) die("store: zla wartosc x=%d",x);
    if (x>sp-1) die("za malo elementow na stosie x=%d, sp=%d",x,sp);

    int y=pop();
    stack[sp-x]=y;    
}

void add(int x=0) {
    int y=pop();
    push(x+y);
}

void sub() {
    int x=pop();
    int y=pop();
    push(y-x);
}

void print() {
    int i=pop();
    cout << i << endl;
}

//------------------------------

void show_stack() {
    for (int i=0;i<sp;i++) {
	if (i>0) cout << " ";
	cout << stack[i];
    }
    cout << endl;
}

int run() {
    int i,x;
    i=0;
    sp=stack.size();
    while(i<(int) program.size()) {
//	printf("cmd=%d arg=%d\n",program[i].cmd,program[i].arg);
	switch(program[i].cmd) {
	case CMD_LABEL: break;
	case CMD_GOTO:  
	    i=program[i].arg-1; 
	    break;
	case CMD_IF: 
	    x=pop();
	    if (x>0) i=program[i].arg-1; 
	    break;
	case CMD_PUSH:  push(program[i].arg); break;
	case CMD_POP:   pop(program[i].arg); break;
	case CMD_DUP:   dup(program[i].arg); break;
	case CMD_STORE: store(program[i].arg); break;
	case CMD_ADD:   add(program[i].arg); break;
	
	case CMD_SUB:   sub(); break;
	
	case CMD_PRINT: print(); break;
	case CMD_DEBUG: cout << "DEB: "; show_stack(); break;

	case CMD_VADD: 
	    x=pop();add(x);
	    break;	
	case CMD_VSTORE: 
	    x=pop();store(x);
	    break;
	case CMD_VDUP: 
	    x=pop();dup(x);	
	    break;
	case CMD_HALT:
	    return 0;
	default:
	    die("nieznane polecenie: %d\n",program[i].cmd);
	}
	i++;
    }
    return 0;
}

int main(int argc,char *argv[]) {
    if (argc!=3) usage();
    
    load_program(argv[1]);
    load_stack(argv[2]);
    run();
    show_stack();
}
