//
// Simple Lighting Model
// Copyright (c) Microsoft Corporation. All rights reserved.
//
// Note: This effect file works with EffectEdit.
//


static float EPSILON = 0.001f;
static float bigNumber=9999.0f;

const float maxDistance;
const float receiversLeaf;
const float receiversLeaf2;
const float receiversLeaf3;
const float receiversLeaf4;
const float receiverRaySqRadius;
const float3 listenerPos;


// texture
texture PlaneData;
texture PlaneParamData;
texture LeafData;

// render target textures
texture Rt1Data;
texture Rt2Data;
texture Rt3Data;

// transformations
float4x4 World      : WORLD;
float4x4 View       : VIEW;
float4x4 Projection : PROJECTION;

struct VS_OUTPUT
{
    float4 Pos  : POSITION;    
    float2 Tex : TEXCOORD0;    
};

VS_OUTPUT VS(
    float3 Pos  : POSITION, 
    float3 Norm : NORMAL, 
    float2 Tex  : TEXCOORD0)
{
    VS_OUTPUT Out = (VS_OUTPUT)0;    
    float4x4 WorldView = mul(World, View);        
    float3 P = mul(Pos, WorldView);    
    Out.Pos  = mul(float4(P, 1), Projection);              // position (projected)
    Out.Tex = Tex;
    return Out;
}

sampler SamplerPlaneData = sampler_state
{
    Texture   = (PlaneData);
    MinFilter = POINT;
    MagFilter = POINT;
    MipFilter = POINT;
};
sampler SamplerPlnParams = sampler_state
{
    Texture   = (PlaneParamData);
    MinFilter = POINT;
    MagFilter = POINT;
    MipFilter = POINT;
};
sampler SamplerLeafData = sampler_state
{
    Texture   = (LeafData);
    MinFilter = POINT;
    MagFilter = POINT;
    MipFilter = POINT;
};

sampler StateData1 = sampler_state
{
    Texture   = (Rt1Data);
    MinFilter = POINT;
    MagFilter = POINT;
    MipFilter = POINT;
};
sampler StateData2 = sampler_state
{
    Texture   = (Rt2Data);
    MinFilter = POINT;
    MagFilter = POINT;
    MipFilter = POINT;
};
sampler StateData3 = sampler_state
{
    Texture   = (Rt3Data);
    MinFilter = POINT;
    MagFilter = POINT;
    MipFilter = POINT;
};


struct PS_OUTPUT
{
    float4 state1 : COLOR0;            
    float4 data1 : COLOR1;   
    float4 data2  : COLOR2;    
};

PS_OUTPUT PS(    
    float2 Tex : TEXCOORD0
   )
{   
   PS_OUTPUT psout;
   
    //state1.x    <- actual leaf number
    //state1.y    <- actual plane/portal examined from this leaf (in this iteration)
    //state1.z    <- ray attenuation due to reflections (sum of all wall attenuation cooeficient hit)
    //state1.w    <- distance to plane closest to origin of ray
    
    //data1.xyz    <- ray origin
    //data1.w      <- distance ray have traveled so far (negative if hit sphere)
    
    //data2.xyz    <- ray direction vector
    //data2.w    <- plane index that has shortest distance to ray origin 

    psout.state1 = tex2D(StateData1 , Tex);
    //dont output anything if finished
    //clip(psout.state1.z);
    
    psout.data1 = tex2D(StateData2 , Tex);
    psout.data2 = tex2D(StateData3 , Tex);

    //get leaf data, .x <- begining of plane data for this leaf,
    //                 .y <- number of planes/portals for this leaf
    //                 .xw <- unused
    float4 leaf_data = tex1D(SamplerLeafData , psout.state1.x*1.0f/2048.0f + 0.5f/2048.0f);
      
    float pos;
    float4 plane;
    float4 params;
    
    float4 plane2;
    float4 params2;
        
        
    if ( psout.state1.y >= leaf_data.y ) 
    {
        pos = psout.data2.w;    //all planes for this leaf were checked so, use index of the one stored
                        // in data2.w <- the closer one plane 
    }
    else
    {
        pos = psout.state1.y; //otherwise take next plane to examine how close it is for current ray
    }

    pos += leaf_data.x;
    pos = pos*1.0f/2048.0f + 0.5f/2048.0f ;

    plane = tex1D(SamplerPlaneData, pos);// plane.xyz - plane normal, plane.w distance from origin
    
    //params.x <- equals 1 if its a plane, 0 if its a portal
    //params.y <- equals leaf number that should be set for next iteration,
    //             if its plane then this ray will stay in the same leaf
    //             if it was portal then this value will be leaf number of neighbouring leaf
    //                where ray should now travel
    //params.z <- equals attenuation coeficient of surface
    params = tex1D(SamplerPlnParams, pos); 
    
    /*
    pos += 1.0f/2048.0f;
    params2 = tex1D(SamplerPlnParams, pos); 
    plane2 = tex1D(SamplerPlaneData, pos);// plane.xyz - plane normal, plane.w distance from origin
    
    plane += plane2;
    params += params2;    
    */
    
    if ( psout.state1.z < 0 )
    {
        //psout.state1.z-=4;
    }
    else    
    if ( psout.state1.y >= leaf_data.y )// if (currentLeaf >= maxPlanesForThisLeaf)
    {
        //now we have found closest plane/portal
        
        //first check if receiver have been struckt with this ray (just test for it)
        float s=0;               
        if ( psout.state1.x == receiversLeaf || psout.state1.x == receiversLeaf2 ) 
        {                                                      // more on this code in Real-Time rendering page.571
            float3 L = listenerPos - psout.data1.xyz;    // vector from ray origin to the sphere position
            s = dot(L, psout.data2.xyz);             // distance to the plane containing center of sphere
            float S2 = s*s;            
            float L2 = dot(L,L);                   // squared distance to center of sphere
            if ( s < 0 && L2 > receiverRaySqRadius)// no intersection if ray points away from sphere
                s = 0;                               //   and is not inside it
            float m2 = L2 - S2;                       // from Pythagorean theorem
            if ( m2 > receiverRaySqRadius )           // 
                s = 0;                               //
            /*
            float q = sqrt(receiverRaySqRadius - m2);
            if ( L2 > receiverRaySqRadius)
                s = s - q;
            else
                s = s + q;
            */
        }
        
        if ( s > 0 )
        {
            psout.data1.w = s; //distance to closest plane     
            psout.state1.z*=-1;       
            params.z = 1;
        }
        
        //we have tested all planes for this leaf so, use the closest as intersecting one
        psout.state1.x = params.y;                //leaf number - use the one from plane data
        psout.state1.y = 0;                       //set tested plane count to zero                
        psout.state1.z *= params.z;     //add plane attenuation cooeficient (zero for portals)        
        psout.data1.xyz += psout.data2.xyz*psout.data1.w;    //compute intersection point
        
        psout.state1.w+= psout.data1.w;    //current distance this ray have traveled 
                        
        psout.data1.w = bigNumber;                //set distance to closest plane to some big number                        
                                
        if ( params.x )                            //if it is a plane, then reflect                            
            psout.data2.xyz = reflect(psout.data2.xyz, plane.xyz); //reflect ray direction        
        
        //psout.data2.w=0;

        //if receiver sphere hit - then stop ray propagation            
        //if reached max reflections then stop propagataion
        //if ray is too weak then also stop propagation
                
        //if (s>=0)
            //psout.state1.z+=1;
            /*
        if (psout.state1.w > maxDistance)
        {
            psout.state1.z+=2;                     
            //psout.state1.w*=-1;                             
        }
        */
            
        //if (psout.state1.w > maxDistance)
        //    psout.state1.z+=10;                     
        
        //if (psout.state1.w > maxDistance)
        //    psout.state1.z*=-1;             
    }
    else
    {
        //there are stil more not yet tested planes       
        float dist;
        
        dist = plane.w - dot(plane.xyz, psout.data1.xyz);
        float dotres = dot (plane.xyz, psout.data2.xyz);
        dist /= dotres;
                                    
        if ( dotres<0 /*&&  dist > EPSILON */&& dist < psout.data1.w )
        {
            psout.data1.w = dist;
            psout.data2.w = psout.state1.y;
        }                                 
        
        psout.state1.y++;
        
    }

    return psout;

}



PS_OUTPUT PS_LeafPlaneIntersect(    
    float2 Tex : TEXCOORD0
   )
{   
   PS_OUTPUT psout;//=0;
    
    //state1.x    <- distance ray have traveled so far (negative if hit sphere)
    //state1.y    <- plane index that has shortest distance to ray origin 
    //state1.z    <- ray attenuation due to reflections (sum of all wall attenuation cooeficient hit)
    //state1.w    <- distance to plane closest to origin of ray
    
    //data1.xyz    <- ray origin
    //data1.w      <- actual leaf number
    
    //data2.xyz    <- ray direction vector
    //data2.w      <- actual plane/portal examined from this leaf (in this iteration)

    psout.state1 = tex2D(StateData1 , Tex);
    //clip(psout);
    //dont output anything if finished
    //clip(psout);
    
    psout.data1 = tex2D(StateData2 , Tex);
    psout.data2 = tex2D(StateData3 , Tex);

    //get leaf data, .x <- begining of plane data for this leaf, (its already multiplied by 1.0f/2048.0f
    //                        and added 0.5f/2048.0f)
    //                 .y <- number of planes/portals for this leaf
    //                 .xw <- unused
    //float4 leaf_data = tex1D(SamplerLeafData , data1.w*1.0f/2048.0f + 0.5f/2048.0f);
    //float pos = leaf_data.x+data2.w*1.0f/2048.0f;
    float pos = psout.data2.w*1.0f/2048.0f + 0.5f/2048.0f;
            
    
    float dist;
    //float old_dist = psout.x;
        
    for (  float i=-1; i>=-6; i--)
    {    
        float4 plane = tex1D(SamplerPlaneData, pos);// plane.xyz - plane normal, plane.w distance from origin
                        
        dist = plane.w - dot(plane.xyz, psout.data1.xyz);
        float dotres = dot (plane.xyz, psout.data2.xyz);
        dist /= dotres;
        float outp=99999;
                
        if ( dotres<0 )
            outp = dist;        
        if ( outp < psout.state1.x )                 
            psout.state1.y = i;
        if ( outp < psout.state1.x )         
            psout.state1.x = outp;
            
        pos += 1.0f/2048.0f;
    }

	
	//if ( psout.state1.y < 0)
	//	psout.state1.y=psout.data2.w+(-psout.state1.y-1);
	
	
	//psout.y += psout.y < 100 ? data1.w : 0;
    return psout;

}

PS_OUTPUT PS_Propagate(    
    float2 Tex : TEXCOORD0
   )
{   

   PS_OUTPUT psout;
   
    //state1.x    <- actual leaf number
    //state1.y    <- actual plane/portal examined from this leaf (in this iteration)
    //state1.z    <- ray attenuation due to reflections (sum of all wall attenuation cooeficient hit)
    //state1.w    <- distance to plane closest to origin of ray
    
    //data1.xyz    <- ray origin
    //data1.w      <- distance ray have traveled so far (negative if hit sphere)
    
    //data2.xyz    <- ray direction vector
    //data2.w    <- plane index that has shortest distance to ray origin 

    psout.state1 = tex2D(StateData1 , Tex);
    //dont output anything if finished
    //clip(psout.state1);
    //clip(psout.state1);
    
    psout.data1 = tex2D(StateData2 , Tex);
    psout.data2 = tex2D(StateData3 , Tex);

    //get leaf data, .x <- begining of plane data for this leaf,
    //                 .y <- number of planes/portals for this leaf
    //                 .xw <- unused
      
    float pos;
    float4 plane;
    float4 params;
    
    pos = psout.state1.y*1.0f/2048.0f;    //all planes for this leaf were checked so, use index of the one stored
    
    float4 leaf_data = tex1D(SamplerLeafData , psout.data1.w*1.0f/2048.0f + 0.5f/2048.0f);      
    //pos += leaf_data.x;
    //pos = pos*1.0f/2048.0f + 0.5f/2048.0f ;

    plane = tex1D(SamplerPlaneData, pos);// plane.xyz - plane normal, plane.w distance from origin
    
    //params.x <- equals 1 if its a plane, 0 if its a portal
    //params.y <- equals leaf number that should be set for next iteration,
    //             if its plane then this ray will stay in the same leaf
    //             if it was portal then this value will be leaf number of neighbouring leaf
    //                where ray should now travel
    //params.z <- equals attenuation coeficient of surface
    params = tex1D(SamplerPlnParams, pos); 
    
    /*
    pos += 1.0f/2048.0f;
    params2 = tex1D(SamplerPlnParams, pos); 
    plane2 = tex1D(SamplerPlaneData, pos);// plane.xyz - plane normal, plane.w distance from origin
    
    plane += plane2;
    params += params2;    
    */
    
    if ( psout.state1.z < 0 )
    {    
		psout.data2.w = -100;
    }
    
    if ( psout.data2.w+6 < leaf_data.y )
    {        
		psout.data2.w += 6;
    }
    else
    {
        //now we have found closest plane/portal
        
        //first check if receiver have been struckt with this ray (just test for it)
        float s=0;               
        
        float test = psout.data1.w - receiversLeaf;
        test *= psout.data1.w - receiversLeaf2;
        test *= psout.data1.w - receiversLeaf3;
        test *= psout.data1.w - receiversLeaf4;
        
        if ( test == 0)   
        {                                                      // more on this code in Real-Time rendering page.571
            float3 L = listenerPos - psout.data1.xyz;    // vector from ray origin to the sphere position
            s = dot(L, psout.data2.xyz);             // distance to the plane containing center of sphere
            float S2 = s*s;            
            float L2 = dot(L,L);                   // squared distance to center of sphere
            if ( s < 0 && L2 > receiverRaySqRadius)// no intersection if ray points away from sphere
                s = 0;                               //   and is not inside it
            float m2 = L2 - S2;                       // from Pythagorean theorem
            if ( m2 > receiverRaySqRadius )           // 
                s = 0;                               //
            
            /*
            float q = sqrt(receiverRaySqRadius - m2);
            if ( s != 0 )
            {
            if ( L2 > receiverRaySqRadius)
                s = s - q;
            else
                s = s + q;
            } 
            */                           
        }
                        
        if ( s > 0 )
        {
            psout.state1.x = s; //distance to closest plane     
            psout.state1.z*=-1;       
            params.z = 1;
            params.x = 0;
        }
        
        //we have tested all planes for this leaf so, use the closest as intersecting one
        psout.data1.w = params.y;                //leaf number - use the one from plane data
        
        float4 new_leaf_data = tex1D(SamplerLeafData , params.y*1.0f/2048.0f + 0.5f/2048.0f);      
        psout.data2.w = (new_leaf_data.x - 0.5f/2048.0f) * 2048.0f; //set tested plane count to zero                        
        
        psout.state1.z *= params.z;     //add plane attenuation cooeficient (zero for portals)        
        psout.data1.xyz += psout.data2.xyz*psout.state1.x;    //compute intersection point
        
        psout.state1.w+= psout.state1.x;    //current distance this ray have traveled 
                        
        psout.state1.x = bigNumber;                //set distance to closest plane to some big number                        
                                
        if ( params.x )                            //if it is a plane, then reflect                            
            psout.data2.xyz = reflect(psout.data2.xyz, plane.xyz); //reflect ray direction        
        
        //psout.data2.w=0;

        //if receiver sphere hit - then stop ray propagation            
        //if reached max reflections then stop propagataion
        //if ray is too weak then also stop propagation
                
        //if (s>=0)
            //psout.state1.z+=1;
        /*
        if (psout.state1.w > maxDistance)
        {        
            psout.state1.w*=-1;
        }
        */
        
    }       
    return psout;

}

float4 PS_MakeOnScreenTexture(    
    float2 Tex : TEXCOORD0
   ) : COLOR
{   
   PS_OUTPUT ps_in;
   float4 out_col=0;
    
   ps_in.state1 = tex2D(StateData1 , Tex);    
   ps_in.data1 = tex2D(StateData2 , Tex);
   ps_in.data2 = tex2D(StateData3 , Tex);

    float refs = log(abs(ps_in.state1.z))/log(0.7);        

    if ( ps_in.state1.z < 0 )
    {
        if ( refs == 0 )
            out_col.r = 1;
        else
        if ( abs(refs - 1)< 0.1f)
            out_col.g = 1;
        else    
        if ( abs(refs - 2)< 0.1f)
            out_col.b = 1;
        else
        //if ( refs > 2 )
            out_col = 1.0f-(refs*10.0f)/255.0f;
    }
            
    return out_col;

}


float4 PS_PackTo32BitTexture (
    float2 Tex : TEXCOORD0
   ) : COLOR
{   
	float4 psout=0;
	float4 state1 = tex2D(StateData1 , Tex);
	
	psout.x = 1.0f;	
	psout.x *= abs(state1.z);
	psout.x += floor(state1.w);
	psout.x *= sign(state1.z);
	
	/*
    float4 res=0;
    float4 state;
    float fpow=pow(2,16);
    for (int i=0; i<8; i++)
    {   
        Tex.x+=1.0f/256.0f;
        state = 1;//tex2D(StateData1 , Tex);
        
        if ( state.z < 0 )
            res.y+=fpow;
        
        fpow*=2;
    }    
*/
    return psout;
}

float4 PS_ComputeImageSources (
    float2 Tex : TEXCOORD0
   ) : COLOR
{   
	PS_OUTPUT ps_in;
   float4 out_col=0;
    
   ps_in.state1 = tex2D(StateData1 , Tex);    
   
   //if attenuated out, to long ray propagation
   //clip(ps_in.state1.w);
   //if this ray have not intersected with receiver
   //clip(-ps_in.state1.z);
   
   ps_in.data1 = tex2D(StateData2 , Tex); //origin point
   ps_in.data2 = tex2D(StateData3 , Tex); //ray direction vector

   //invert it
   ps_in.data2 *= -1.0f;
   ps_in.data1 += ps_in.state1.w * ps_in.data2;
    
   out_col.xyz = ps_in.data1.xyz; //sound image source position
   out_col.w = ps_in.state1.z;   //attenuation due to reflections
            
   return out_col;
}


technique TVertexAndPixelShader
{
    pass P0
    {
        // shaders
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS();
    }  
}

technique LeafPlaneIntersect
{
    pass P0
    {
        // shaders
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS_LeafPlaneIntersect();
    }  
}

technique ATIRadeon9800
{
   pass P0
    {
        // shaders
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS_LeafPlaneIntersect();
        //PixelShader  = compile ps_2_0 PS_ComputeImageSources();
    }  
   pass P1
    {
        // shaders
        VertexShader = compile vs_2_0 VS();        
        PixelShader  = compile ps_2_0 PS_Propagate();
        
        //PixelShader  = compile ps_2_0 PS_LeafPlaneIntersect();
        //PixelShader  = compile ps_2_0 PS_ComputeImageSources();
    }  
    pass P2
    {
        // shaders
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS_PackTo32BitTexture();
    }      
    
    /*
    this pass should be computed once after all propagations
    */    
    pass P3
    {
        // shaders
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS_ComputeImageSources();
    }          
}

technique MakeOnScreenTexture
{
  pass P0
    {
        // shaders
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS_MakeOnScreenTexture();
    }  
}

technique RayPropagate
{
    pass P0
    {
        // shaders
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS_Propagate();
    }  
}

technique PackTo32BitTexture
{
    pass P0
    {
        // shaders
        VertexShader = compile vs_2_0 VS();
        PixelShader  = compile ps_2_0 PS_PackTo32BitTexture();
    }      
}